<?php

// --- Configuration ---
$trades_file = 'trades.json';
// $success_message = ''; // Not used due to redirect
$error_message = '';

// --- Helper Functions ---

// Mapping for Font Awesome icons - Handles pairs, specific tickers, and defaults
function getIconClassForPair(string $assetString): string {
    // Mapping for BASE assets in PAIRS (Crypto, Forex, Commodities)
    $pairBaseIconMapping = [ /* Same as previous version */
        "BTC" => "fab fa-btc btc-icon", "ETH" => "fab fa-ethereum eth-icon",
        "SOL" => "fas fa-sun sol-icon", "DOGE"=> "fas fa-paw doge-icon",
        "ADA" => "fas fa-leaf ada-icon", "XRP" => "fas fa-circle-notch xrp-icon",
        "LTC" => "fas fa-bolt ltc-icon", "AUD" => "fas fa-dollar-sign aud-icon",
        "EUR" => "fas fa-euro-sign eur-icon", "GBP" => "fas fa-pound-sign gbp-icon",
        "JPY" => "fas fa-yen-sign jpy-icon", "USD" => "fas fa-dollar-sign usd-icon",
        "CAD" => "fas fa-dollar-sign cad-icon", "CHF" => "fas fa-franc-sign chf-icon",
        "XAU" => "fas fa-cube xau-icon", "XAG" => "fas fa-ring xag-icon",
        "WTI" => "fas fa-tint oil-icon", "OIL" => "fas fa-tint oil-icon",
    ];
    // Specific mapping for known single Tickers (Stocks, Indices)
    $tickerIconMapping = [ /* Same as previous version + SPX */
         "AAPL"=> "fab fa-apple", "MSFT"=> "fab fa-microsoft",
         "GOOGL"=> "fab fa-google", "AMZN"=> "fab fa-amazon",
         "META"=> "fab fa-facebook",
         "SPX" => "fas fa-stream", // Added icon for S&P 500 Index
         // Add other indices like "NDX" (Nasdaq 100), "DJI" (Dow Jones) if needed
         // "NDX" => "fas fa-stream",
         // "DJI" => "fas fa-stream",
    ];

    $genericStockIcon = "fas fa-briefcase"; // Generic icon for most stocks/unmapped tickers
    $defaultIcon = "fas fa-chart-line"; // Default for unrecognized formats

    if (empty($assetString)) {
        return $defaultIcon;
    }

    $assetString = trim($assetString);
    $assetUpper = strtoupper($assetString);

    // Check if it's likely a pair (contains '/')
    if (strpos($assetString, '/') !== false) {
        // Pair logic (same as before)
        $parts = explode('/', $assetString);
        $baseAsset = strtoupper(trim($parts[0]));
        return $pairBaseIconMapping[$baseAsset] ?? $defaultIcon;
    } else {
        // Single ticker (Stock, Index, etc.) logic
        // Check specific ticker map first (now includes SPX)
        if (isset($tickerIconMapping[$assetUpper])) {
            return $tickerIconMapping[$assetUpper];
        }
        // Otherwise, return the generic stock icon
        return $genericStockIcon;
    }
}


// Function to read trades from JSON file (same as before)
function readTrades(string $filename): array {
    if (!file_exists($filename)) return [];
    $json_data = @file_get_contents($filename);
    if ($json_data === false) { error_log("Failed to read trades file: " . $filename); return []; }
    $trades = json_decode($json_data, true);
    if (json_last_error() !== JSON_ERROR_NONE) { error_log("JSON decode error: " . json_last_error_msg() . " in file: " . $filename); return []; }
    return is_array($trades) ? $trades : [];
}

// Function to write trades to JSON file (same as before)
function writeTrades(string $filename, array $trades): bool {
    $json_data = json_encode($trades, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    if ($json_data === false) { error_log("JSON encode error: " . json_last_error_msg()); return false; }
    $result = @file_put_contents($filename, $json_data, LOCK_EX);
     if ($result === false) { error_log("Failed to write trades file (check permissions): " . $filename); return false; }
    return true;
}


// --- Handle Form Submission (POST Request) ---
// The POST handling logic remains exactly the same. It will correctly
// read "SPX" from the form and pass it to the updated getIconClassForPair.
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validation logic (same as before)...
    $required_fields = ['pair', 'status', 'roi', 'entryPrice', 'pnl'];
    $missing_fields = [];
    foreach ($required_fields as $field) { if (!filter_has_var(INPUT_POST, $field) || trim($_POST[$field]) === '') { $missing_fields[] = ucfirst(preg_replace('/(?<!^)[A-Z]/', ' $0', $field)); } }
    if ($_POST['status'] === 'Closed' && trim($_POST['exitPrice']) === '') { /* Optional check */ }

    if (!empty($missing_fields)) {
        $error_message = "Missing required fields: " . implode(', ', $missing_fields);
    } else {
        // Process valid data using filter_input (same as before)...
        $roi = filter_input(INPUT_POST, 'roi', FILTER_VALIDATE_FLOAT);
        $entryPrice = filter_input(INPUT_POST, 'entryPrice', FILTER_VALIDATE_FLOAT);
        $exitPrice = filter_input(INPUT_POST, 'exitPrice', FILTER_VALIDATE_FLOAT, FILTER_NULL_ON_FAILURE);
        $pnl = filter_input(INPUT_POST, 'pnl', FILTER_VALIDATE_FLOAT);
        $pair = filter_input(INPUT_POST, 'pair', FILTER_SANITIZE_STRING); // Reads 'SPX', 'AAPL' or 'BTC/USDT' etc.
        $leverage = filter_input(INPUT_POST, 'leverage', FILTER_SANITIZE_STRING);
        $status = filter_input(INPUT_POST, 'status', FILTER_SANITIZE_STRING);

        if ($roi === false || $entryPrice === false || $pnl === false) { $error_message = "Invalid number format for ROI, Entry Price, or P&L."; }
        elseif ($exitPrice === false && !empty($_POST['exitPrice'])) { $error_message = "Invalid number format for Exit Price."; }
        else {
             $tradeId = time() . rand(100, 999);
             $newTrade = [ /* Create $newTrade array same as before, using the variables above */
                'id' => $tradeId, 'pair' => $pair, 'leverage' => !empty($leverage) ? $leverage : null,
                'status' => $status, 'roi' => $roi, 'entryPrice' => $entryPrice, 'exitPrice' => $exitPrice, 'pnl' => $pnl,
                'dateOpened' => !empty($_POST['dateOpened']) ? (new DateTime($_POST['dateOpened']))->format(DateTime::ATOM) : null,
                'dateClosed' => ($status === 'Closed' && !empty($_POST['dateClosed'])) ? (new DateTime($_POST['dateClosed']))->format(DateTime::ATOM) : (($status === 'Closed') ? (new DateTime())->format(DateTime::ATOM) : null),
                'iconClass' => getIconClassForPair($pair), // Uses the updated function
                'lastUpdated' => (new DateTime())->format(DateTime::ATOM) ];

            $trades = readTrades($trades_file); $trades[] = $newTrade;
            if (writeTrades($trades_file, $trades)) { header("Location: display_trade.php?id=" . urlencode($tradeId)); exit; }
            else { $error_message = "Error saving trade data. Check server logs and file permissions for '{$trades_file}'."; }
        }
    }
}
// --- Display Form (GET Request or Failed POST) ---
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin - Add Trade</title>
    <style> /* CSS Styles remain the same */ </style>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif; line-height: 1.6; padding: 20px; background-color: #f8f9fa; color: #212529; } .container { max-width: 650px; margin: 20px auto; background: #ffffff; padding: 30px; border-radius: 8px; box-shadow: 0 4px 12px rgba(0,0,0,0.08); border: 1px solid #dee2e6; } h1 { text-align: center; color: #343a40; margin-bottom: 25px; font-weight: 500; } .form-group { margin-bottom: 18px; } label { display: block; margin-bottom: 6px; font-weight: 500; color: #495057; font-size: 0.95em;} input[type="text"], input[type="number"], input[type="datetime-local"], select { width: 100%; padding: 10px 12px; border: 1px solid #ced4da; border-radius: 4px; box-sizing: border-box; font-size: 1em; transition: border-color 0.15s ease-in-out, box-shadow 0.15s ease-in-out; } input:focus, select:focus { border-color: #80bdff; outline: 0; box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25); } input[type="number"] { appearance: textfield; } button[type="submit"] { background-color: #28a745; color: white; padding: 12px 20px; border: none; border-radius: 5px; cursor: pointer; font-size: 16px; font-weight: 500; width: 100%; margin-top: 15px; transition: background-color 0.2s; } button[type="submit"]:hover { background-color: #218838; } .currency-input { display: flex; align-items: center; } .currency-input span { padding: 10px 12px; background-color: #e9ecef; border: 1px solid #ced4da; border-right: none; border-radius: 4px 0 0 4px; color: #495057;} .currency-input input { border-radius: 0 4px 4px 0; } .message { border: 1px solid transparent; padding: 15px; margin-bottom: 20px; border-radius: 4px; text-align: center; font-weight: 500; } .error-message { background-color: #f8d7da; color: #721c24; border-color: #f5c6cb; }
    </style>
</head>
<body>
    <div class="container">
        <h1>Add New Trade Record</h1>

        <?php if (!empty($error_message)): ?>
            <div class="message error-message"><?php echo htmlspecialchars($error_message); ?></div>
        <?php endif; ?>

        <form action="admin.php" method="POST">
            <div class="form-group">
                <label for="pair">Asset/Pair:</label>
                <select id="pair" name="pair" required>
                    <option value="">-- Select Asset --</option>
                     <optgroup label="Indices">
                        <option value="SPX">SPX (S&P 500 Index)</option>
                        </optgroup>
                    <optgroup label="Stocks">
                        <option value="AAPL">AAPL (Apple)</option>
                        <option value="MSFT">MSFT (Microsoft)</option>
                        <option value="GOOGL">GOOGL (Alphabet/Google)</option>
                        <option value="AMZN">AMZN (Amazon)</option>
                        <option value="TSLA">TSLA (Tesla)</option>
                        <option value="NVDA">NVDA (NVIDIA)</option>
                        <option value="META">META (Meta/Facebook)</option>
                        <option value="JPM">JPM (JPMorgan Chase)</option>
                        <option value="V">V (Visa)</option>
                        <option value="WMT">WMT (Walmart)</option>
                    </optgroup>
                     <optgroup label="Cryptocurrencies">
                        <option value="BTC/USDT">BTC/USDT</option>
                        <option value="ETH/USDT">ETH/USDT</option>
                        <option value="SOL/USDT">SOL/USDT</option>
                        <option value="DOGE/USDT">DOGE/USDT</option>
                        <option value="ADA/USDT">ADA/USDT</option>
                        <option value="XRP/USDT">XRP/USDT</option>
                        <option value="LTC/USDT">LTC/USDT</option>
                        <option value="BTC/USD">BTC/USD</option>
                        <option value="ETH/USD">ETH/USD</option>
                    </optgroup>
                     <optgroup label="Forex">
                        <option value="EUR/USD">EUR/USD</option>
                        <option value="GBP/USD">GBP/USD</option>
                        <option value="USD/JPY">USD/JPY</option>
                        <option value="AUD/USD">AUD/USD</option>
                        <option value="USD/CAD">USD/CAD</option>
                        <option value="USD/CHF">USD/CHF</option>
                        <option value="EUR/GBP">EUR/GBP</option>
                        <option value="EUR/JPY">EUR/JPY</option>
                        <option value="GBP/JPY">GBP/JPY</option>
                     </optgroup>
                     <optgroup label="Commodities">
                        <option value="XAU/USD">XAU/USD (Gold)</option>
                        <option value="XAG/USD">XAG/USD (Silver)</option>
                        <option value="WTI/USD">WTI/USD (Oil)</option>
                        <option value="OIL/USD">OIL/USD (Oil Alias)</option>
                     </optgroup>
                </select>
            </div>

            <div class="form-group">
                <label for="leverage">Leverage & Direction:</label>
                <input type="text" id="leverage" name="leverage" placeholder="e.g., Long 10X (Less common for indices/stocks)">
            </div>

            <div class="form-group"> <label for="status">Status:</label> <select id="status" name="status" required> <option value="Open">Open</option> <option value="Closed">Closed</option> </select> </div>
             <div class="form-group"> <label for="roi">ROI (%):</label> <input type="number" step="any" id="roi" name="roi" placeholder="e.g., 15.50 or -5.20" required> </div>
             <div class="form-group"> <label for="entryPrice">Entry Price:</label> <input type="number" step="any" id="entryPrice" name="entryPrice" placeholder="e.g., 4500.75" required> </div>
             <div class="form-group"> <label for="exitPrice">Exit/Current Price:</label> <input type="number" step="any" id="exitPrice" name="exitPrice" placeholder="Enter exit price if Closed"> </div>
             <div class="form-group"> <label for="pnl">Profit & Loss (P&L):</label> <div class="currency-input"> <span>$</span> <input type="number" step="any" id="pnl" name="pnl" placeholder="e.g., 1230.00 or -300.00" required> </div> </div>
             <div class="form-group"> <label for="dateOpened">Date Opened:</label> <input type="datetime-local" id="dateOpened" name="dateOpened" title="Optional - When the trade was initiated"> </div>
             <div class="form-group"> <label for="dateClosed">Date Closed:</label> <input type="datetime-local" id="dateClosed" name="dateClosed" title="Leave blank if status is 'Open'. If left blank for 'Closed', current time will be used."> </div>
            <button type="submit">Add Trade</button>

        </form>
    </div>
</body>
</html>